/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.linear;

import java.util.Arrays;
import org.apache.commons.math3.exception.DimensionMismatchException;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.BlockRealMatrix;
import org.apache.commons.math3.linear.DecompositionSolver;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.apache.commons.math3.linear.SingularMatrixException;
import org.apache.commons.math3.util.FastMath;

public class QRDecomposition {
    private double[][] qrt;
    private double[] rDiag;
    private RealMatrix cachedQ;
    private RealMatrix cachedQT;
    private RealMatrix cachedR;
    private RealMatrix cachedH;
    private final double threshold;

    public QRDecomposition(RealMatrix matrix) {
        this(matrix, 0.0);
    }

    public QRDecomposition(RealMatrix matrix, double threshold) {
        this.threshold = threshold;
        int m2 = matrix.getRowDimension();
        int n2 = matrix.getColumnDimension();
        this.qrt = matrix.transpose().getData();
        this.rDiag = new double[FastMath.min(m2, n2)];
        this.cachedQ = null;
        this.cachedQT = null;
        this.cachedR = null;
        this.cachedH = null;
        this.decompose(this.qrt);
    }

    protected void decompose(double[][] matrix) {
        for (int minor = 0; minor < FastMath.min(matrix.length, matrix[0].length); ++minor) {
            this.performHouseholderReflection(minor, matrix);
        }
    }

    protected void performHouseholderReflection(int minor, double[][] matrix) {
        double a2;
        double[] qrtMinor = matrix[minor];
        double xNormSqr = 0.0;
        for (int row = minor; row < qrtMinor.length; ++row) {
            double c2 = qrtMinor[row];
            xNormSqr += c2 * c2;
        }
        this.rDiag[minor] = a2 = qrtMinor[minor] > 0.0 ? -FastMath.sqrt(xNormSqr) : FastMath.sqrt(xNormSqr);
        if (a2 != 0.0) {
            int n2 = minor;
            qrtMinor[n2] = qrtMinor[n2] - a2;
            for (int col = minor + 1; col < matrix.length; ++col) {
                int row;
                double[] qrtCol = matrix[col];
                double alpha = 0.0;
                for (row = minor; row < qrtCol.length; ++row) {
                    alpha -= qrtCol[row] * qrtMinor[row];
                }
                alpha /= a2 * qrtMinor[minor];
                for (row = minor; row < qrtCol.length; ++row) {
                    int n3 = row;
                    qrtCol[n3] = qrtCol[n3] - alpha * qrtMinor[row];
                }
            }
        }
    }

    public RealMatrix getR() {
        if (this.cachedR == null) {
            int n2 = this.qrt.length;
            int m2 = this.qrt[0].length;
            double[][] ra = new double[m2][n2];
            for (int row = FastMath.min(m2, n2) - 1; row >= 0; --row) {
                ra[row][row] = this.rDiag[row];
                for (int col = row + 1; col < n2; ++col) {
                    ra[row][col] = this.qrt[col][row];
                }
            }
            this.cachedR = MatrixUtils.createRealMatrix(ra);
        }
        return this.cachedR;
    }

    public RealMatrix getQ() {
        if (this.cachedQ == null) {
            this.cachedQ = this.getQT().transpose();
        }
        return this.cachedQ;
    }

    public RealMatrix getQT() {
        if (this.cachedQT == null) {
            int minor;
            int n2 = this.qrt.length;
            int m2 = this.qrt[0].length;
            double[][] qta = new double[m2][m2];
            for (minor = m2 - 1; minor >= FastMath.min(m2, n2); --minor) {
                qta[minor][minor] = 1.0;
            }
            for (minor = FastMath.min(m2, n2) - 1; minor >= 0; --minor) {
                double[] qrtMinor = this.qrt[minor];
                qta[minor][minor] = 1.0;
                if (qrtMinor[minor] == 0.0) continue;
                for (int col = minor; col < m2; ++col) {
                    int row;
                    double alpha = 0.0;
                    for (row = minor; row < m2; ++row) {
                        alpha -= qta[col][row] * qrtMinor[row];
                    }
                    alpha /= this.rDiag[minor] * qrtMinor[minor];
                    for (row = minor; row < m2; ++row) {
                        double[] dArray = qta[col];
                        int n3 = row;
                        dArray[n3] = dArray[n3] + -alpha * qrtMinor[row];
                    }
                }
            }
            this.cachedQT = MatrixUtils.createRealMatrix(qta);
        }
        return this.cachedQT;
    }

    public RealMatrix getH() {
        if (this.cachedH == null) {
            int n2 = this.qrt.length;
            int m2 = this.qrt[0].length;
            double[][] ha = new double[m2][n2];
            for (int i2 = 0; i2 < m2; ++i2) {
                for (int j2 = 0; j2 < FastMath.min(i2 + 1, n2); ++j2) {
                    ha[i2][j2] = this.qrt[j2][i2] / -this.rDiag[j2];
                }
            }
            this.cachedH = MatrixUtils.createRealMatrix(ha);
        }
        return this.cachedH;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.qrt, this.rDiag, this.threshold);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] qrt;
        private final double[] rDiag;
        private final double threshold;

        private Solver(double[][] qrt, double[] rDiag, double threshold) {
            this.qrt = qrt;
            this.rDiag = rDiag;
            this.threshold = threshold;
        }

        public boolean isNonSingular() {
            for (double diag : this.rDiag) {
                if (!(FastMath.abs(diag) <= this.threshold)) continue;
                return false;
            }
            return true;
        }

        public RealVector solve(RealVector b2) {
            int n2 = this.qrt.length;
            int m2 = this.qrt[0].length;
            if (b2.getDimension() != m2) {
                throw new DimensionMismatchException(b2.getDimension(), m2);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            double[] x2 = new double[n2];
            double[] y2 = b2.toArray();
            for (int minor = 0; minor < FastMath.min(m2, n2); ++minor) {
                int row;
                double[] qrtMinor = this.qrt[minor];
                double dotProduct = 0.0;
                for (row = minor; row < m2; ++row) {
                    dotProduct += y2[row] * qrtMinor[row];
                }
                dotProduct /= this.rDiag[minor] * qrtMinor[minor];
                for (row = minor; row < m2; ++row) {
                    int n3 = row;
                    y2[n3] = y2[n3] + dotProduct * qrtMinor[row];
                }
            }
            for (int row = this.rDiag.length - 1; row >= 0; --row) {
                int n4 = row;
                y2[n4] = y2[n4] / this.rDiag[row];
                double yRow = y2[row];
                double[] qrtRow = this.qrt[row];
                x2[row] = yRow;
                for (int i2 = 0; i2 < row; ++i2) {
                    int n5 = i2;
                    y2[n5] = y2[n5] - yRow * qrtRow[i2];
                }
            }
            return new ArrayRealVector(x2, false);
        }

        public RealMatrix solve(RealMatrix b2) {
            int n2 = this.qrt.length;
            int m2 = this.qrt[0].length;
            if (b2.getRowDimension() != m2) {
                throw new DimensionMismatchException(b2.getRowDimension(), m2);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int columns = b2.getColumnDimension();
            int blockSize = 52;
            int cBlocks = (columns + 52 - 1) / 52;
            double[][] xBlocks = BlockRealMatrix.createBlocksLayout(n2, columns);
            double[][] y2 = new double[b2.getRowDimension()][52];
            double[] alpha = new double[52];
            for (int kBlock = 0; kBlock < cBlocks; ++kBlock) {
                int kStart = kBlock * 52;
                int kEnd = FastMath.min(kStart + 52, columns);
                int kWidth = kEnd - kStart;
                b2.copySubMatrix(0, m2 - 1, kStart, kEnd - 1, y2);
                for (int minor = 0; minor < FastMath.min(m2, n2); ++minor) {
                    int k2;
                    double[] yRow;
                    double d2;
                    int row;
                    double[] qrtMinor = this.qrt[minor];
                    double factor = 1.0 / (this.rDiag[minor] * qrtMinor[minor]);
                    Arrays.fill(alpha, 0, kWidth, 0.0);
                    for (row = minor; row < m2; ++row) {
                        d2 = qrtMinor[row];
                        yRow = y2[row];
                        for (k2 = 0; k2 < kWidth; ++k2) {
                            int n3 = k2;
                            alpha[n3] = alpha[n3] + d2 * yRow[k2];
                        }
                    }
                    int k3 = 0;
                    while (k3 < kWidth) {
                        int n4 = k3++;
                        alpha[n4] = alpha[n4] * factor;
                    }
                    for (row = minor; row < m2; ++row) {
                        d2 = qrtMinor[row];
                        yRow = y2[row];
                        for (k2 = 0; k2 < kWidth; ++k2) {
                            int n5 = k2;
                            yRow[n5] = yRow[n5] + alpha[k2] * d2;
                        }
                    }
                }
                for (int j2 = this.rDiag.length - 1; j2 >= 0; --j2) {
                    int jBlock = j2 / 52;
                    int jStart = jBlock * 52;
                    double factor = 1.0 / this.rDiag[j2];
                    double[] yJ = y2[j2];
                    double[] xBlock = xBlocks[jBlock * cBlocks + kBlock];
                    int index = (j2 - jStart) * kWidth;
                    for (int k4 = 0; k4 < kWidth; ++k4) {
                        int n6 = k4;
                        yJ[n6] = yJ[n6] * factor;
                        xBlock[index++] = yJ[k4];
                    }
                    double[] qrtJ = this.qrt[j2];
                    for (int i2 = 0; i2 < j2; ++i2) {
                        double rIJ = qrtJ[i2];
                        double[] yI = y2[i2];
                        for (int k5 = 0; k5 < kWidth; ++k5) {
                            int n7 = k5;
                            yI[n7] = yI[n7] - yJ[k5] * rIJ;
                        }
                    }
                }
            }
            return new BlockRealMatrix(n2, columns, xBlocks, false);
        }

        public RealMatrix getInverse() {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.qrt[0].length));
        }
    }
}

