/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.linear;

import org.apache.commons.math3.exception.DimensionMismatchException;
import org.apache.commons.math3.linear.Array2DRowRealMatrix;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.DecompositionSolver;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.NonSquareMatrixException;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.apache.commons.math3.linear.SingularMatrixException;
import org.apache.commons.math3.util.FastMath;

public class LUDecomposition {
    private static final double DEFAULT_TOO_SMALL = 1.0E-11;
    private final double[][] lu;
    private final int[] pivot;
    private boolean even;
    private boolean singular;
    private RealMatrix cachedL;
    private RealMatrix cachedU;
    private RealMatrix cachedP;

    public LUDecomposition(RealMatrix matrix) {
        this(matrix, 1.0E-11);
    }

    public LUDecomposition(RealMatrix matrix, double singularityThreshold) {
        if (!matrix.isSquare()) {
            throw new NonSquareMatrixException(matrix.getRowDimension(), matrix.getColumnDimension());
        }
        int m2 = matrix.getColumnDimension();
        this.lu = matrix.getData();
        this.pivot = new int[m2];
        this.cachedL = null;
        this.cachedU = null;
        this.cachedP = null;
        for (int row = 0; row < m2; ++row) {
            this.pivot[row] = row;
        }
        this.even = true;
        this.singular = false;
        for (int col = 0; col < m2; ++col) {
            int i2;
            for (int row = 0; row < col; ++row) {
                double[] luRow = this.lu[row];
                double sum = luRow[col];
                for (int i3 = 0; i3 < row; ++i3) {
                    sum -= luRow[i3] * this.lu[i3][col];
                }
                luRow[col] = sum;
            }
            int max = col;
            double largest = Double.NEGATIVE_INFINITY;
            for (int row = col; row < m2; ++row) {
                double[] luRow = this.lu[row];
                double sum = luRow[col];
                for (i2 = 0; i2 < col; ++i2) {
                    sum -= luRow[i2] * this.lu[i2][col];
                }
                luRow[col] = sum;
                if (!(FastMath.abs(sum) > largest)) continue;
                largest = FastMath.abs(sum);
                max = row;
            }
            if (FastMath.abs(this.lu[max][col]) < singularityThreshold) {
                this.singular = true;
                return;
            }
            if (max != col) {
                double tmp = 0.0;
                double[] luMax = this.lu[max];
                double[] luCol = this.lu[col];
                for (i2 = 0; i2 < m2; ++i2) {
                    tmp = luMax[i2];
                    luMax[i2] = luCol[i2];
                    luCol[i2] = tmp;
                }
                int temp = this.pivot[max];
                this.pivot[max] = this.pivot[col];
                this.pivot[col] = temp;
                this.even = !this.even;
            }
            double luDiag = this.lu[col][col];
            for (int row = col + 1; row < m2; ++row) {
                double[] dArray = this.lu[row];
                int n2 = col;
                dArray[n2] = dArray[n2] / luDiag;
            }
        }
    }

    public RealMatrix getL() {
        if (this.cachedL == null && !this.singular) {
            int m2 = this.pivot.length;
            this.cachedL = MatrixUtils.createRealMatrix(m2, m2);
            for (int i2 = 0; i2 < m2; ++i2) {
                double[] luI = this.lu[i2];
                for (int j2 = 0; j2 < i2; ++j2) {
                    this.cachedL.setEntry(i2, j2, luI[j2]);
                }
                this.cachedL.setEntry(i2, i2, 1.0);
            }
        }
        return this.cachedL;
    }

    public RealMatrix getU() {
        if (this.cachedU == null && !this.singular) {
            int m2 = this.pivot.length;
            this.cachedU = MatrixUtils.createRealMatrix(m2, m2);
            for (int i2 = 0; i2 < m2; ++i2) {
                double[] luI = this.lu[i2];
                for (int j2 = i2; j2 < m2; ++j2) {
                    this.cachedU.setEntry(i2, j2, luI[j2]);
                }
            }
        }
        return this.cachedU;
    }

    public RealMatrix getP() {
        if (this.cachedP == null && !this.singular) {
            int m2 = this.pivot.length;
            this.cachedP = MatrixUtils.createRealMatrix(m2, m2);
            for (int i2 = 0; i2 < m2; ++i2) {
                this.cachedP.setEntry(i2, this.pivot[i2], 1.0);
            }
        }
        return this.cachedP;
    }

    public int[] getPivot() {
        return (int[])this.pivot.clone();
    }

    public double getDeterminant() {
        if (this.singular) {
            return 0.0;
        }
        int m2 = this.pivot.length;
        double determinant = this.even ? 1.0 : -1.0;
        for (int i2 = 0; i2 < m2; ++i2) {
            determinant *= this.lu[i2][i2];
        }
        return determinant;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.lu, this.pivot, this.singular);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] lu;
        private final int[] pivot;
        private final boolean singular;

        private Solver(double[][] lu, int[] pivot, boolean singular) {
            this.lu = lu;
            this.pivot = pivot;
            this.singular = singular;
        }

        public boolean isNonSingular() {
            return !this.singular;
        }

        public RealVector solve(RealVector b2) {
            int i2;
            double bpCol;
            int col;
            int m2 = this.pivot.length;
            if (b2.getDimension() != m2) {
                throw new DimensionMismatchException(b2.getDimension(), m2);
            }
            if (this.singular) {
                throw new SingularMatrixException();
            }
            double[] bp = new double[m2];
            for (int row = 0; row < m2; ++row) {
                bp[row] = b2.getEntry(this.pivot[row]);
            }
            for (col = 0; col < m2; ++col) {
                bpCol = bp[col];
                for (i2 = col + 1; i2 < m2; ++i2) {
                    int n2 = i2;
                    bp[n2] = bp[n2] - bpCol * this.lu[i2][col];
                }
            }
            for (col = m2 - 1; col >= 0; --col) {
                int n3 = col;
                bp[n3] = bp[n3] / this.lu[col][col];
                bpCol = bp[col];
                for (i2 = 0; i2 < col; ++i2) {
                    int n4 = i2;
                    bp[n4] = bp[n4] - bpCol * this.lu[i2][col];
                }
            }
            return new ArrayRealVector(bp, false);
        }

        public RealMatrix solve(RealMatrix b2) {
            double[] bpCol;
            int col;
            int m2 = this.pivot.length;
            if (b2.getRowDimension() != m2) {
                throw new DimensionMismatchException(b2.getRowDimension(), m2);
            }
            if (this.singular) {
                throw new SingularMatrixException();
            }
            int nColB = b2.getColumnDimension();
            double[][] bp = new double[m2][nColB];
            for (int row = 0; row < m2; ++row) {
                double[] bpRow = bp[row];
                int pRow = this.pivot[row];
                for (int col2 = 0; col2 < nColB; ++col2) {
                    bpRow[col2] = b2.getEntry(pRow, col2);
                }
            }
            for (col = 0; col < m2; ++col) {
                bpCol = bp[col];
                for (int i2 = col + 1; i2 < m2; ++i2) {
                    double[] bpI = bp[i2];
                    double luICol = this.lu[i2][col];
                    for (int j2 = 0; j2 < nColB; ++j2) {
                        int n2 = j2;
                        bpI[n2] = bpI[n2] - bpCol[j2] * luICol;
                    }
                }
            }
            for (col = m2 - 1; col >= 0; --col) {
                bpCol = bp[col];
                double luDiag = this.lu[col][col];
                int j3 = 0;
                while (j3 < nColB) {
                    int n3 = j3++;
                    bpCol[n3] = bpCol[n3] / luDiag;
                }
                for (int i3 = 0; i3 < col; ++i3) {
                    double[] bpI = bp[i3];
                    double luICol = this.lu[i3][col];
                    for (int j4 = 0; j4 < nColB; ++j4) {
                        int n4 = j4;
                        bpI[n4] = bpI[n4] - bpCol[j4] * luICol;
                    }
                }
            }
            return new Array2DRowRealMatrix(bp, false);
        }

        public RealMatrix getInverse() {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.pivot.length));
        }
    }
}

